#!/usr/bin/env python3
"""
Entry‑point for the loop interference simulation.

This script conforms to the unified simulation interface used throughout
the volume‑4 monorepo. It expects a YAML configuration file and an
output directory. The configuration file provides arrays of parameters
used by the underlying simulation. The heavy lifting is performed by
``scripts.run_interference.run_interference`` which accepts the
parameter arrays and writes its results into the supplied output
directory.

Usage::

    python run.py --config configs/default.yaml --output-dir <dir>

This entry‑point intentionally contains no simulation logic. All
implementation details live in ``scripts/run_interference.py``. If the
configuration file is missing required keys a ``KeyError`` will be
raised.
"""

import argparse
import os
import yaml

from scripts.run_interference import run_interference


def main() -> None:
    parser = argparse.ArgumentParser(description="Run the loop interference simulation")
    parser.add_argument("--config", required=True, help="Path to a YAML configuration file")
    parser.add_argument(
        "--output-dir", required=True, help="Directory in which simulation outputs should be written"
    )
    args = parser.parse_args()

    # Load configuration
    cfg_path = args.config
    with open(cfg_path, "r", encoding="utf-8") as f:
        cfg = yaml.safe_load(f)

    # Determine output directory and ensure it exists
    out_dir = args.output_dir
    os.makedirs(out_dir, exist_ok=True)

    # Extract required parameter arrays from the configuration.  A KeyError
    # will be raised if any expected section is missing.  Angles are
    # nested under the ``interference`` section of the YAML.
    b_values = cfg["b_values"]
    k_values = cfg["k_values"]
    n0_values = cfg["n0_values"]
    L_values = cfg["L_values"]
    angles = cfg.get("interference", {}).get("angles")
    if angles is None:
        raise KeyError("interference.angles must be defined in the configuration file")

    # Run the simulation
    run_interference(
        b_values=b_values,
        k_values=k_values,
        n0_values=n0_values,
        L_values=L_values,
        angles=angles,
        output_dir=out_dir,
    )


if __name__ == "__main__":
    main()